(function () {
    'use strict';

    // AJAX filtering for list shortcodes
    document.querySelectorAll('.enacast-container[data-enacast-filterable]').forEach(initContainer);

    function initContainer(container) {
        var shortcode = container.getAttribute('data-enacast-shortcode');
        var baseAtts = {};
        try {
            baseAtts = JSON.parse(container.getAttribute('data-enacast-atts') || '{}');
        } catch (e) {}

        var form = container.querySelector('.enacast-filter-form');
        var content = container.querySelector('.enacast-content');

        if (!form || !content) return;

        // Intercept form submit
        form.addEventListener('submit', function (e) {
            e.preventDefault();
            fetchFiltered(1);
        });

        // Intercept pagination clicks
        content.addEventListener('click', function (e) {
            var link = e.target.closest('.enacast-pagination-link');
            if (link) {
                e.preventDefault();
                var page = link.getAttribute('data-page');
                if (page) {
                    fetchFiltered(parseInt(page, 10));
                }
            }
        });

        function fetchFiltered(page) {
            var atts = Object.assign({}, baseAtts);
            var formData = new FormData(form);

            // Unchecked checkboxes are not in FormData, so explicitly
            // clear their base defaults when unchecked
            form.querySelectorAll('input[type="checkbox"]').forEach(function (cb) {
                if (!cb.checked) {
                    atts[cb.name] = '';
                }
            });

            formData.forEach(function (value, key) {
                if (value) {
                    atts[key] = value;
                }
            });
            atts.page = page || 1;

            // Show loading state
            container.classList.add('enacast-loading');

            var xhr = new XMLHttpRequest();
            xhr.open('POST', enacast_ajax.ajax_url, true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

            xhr.onload = function () {
                container.classList.remove('enacast-loading');
                try {
                    var resp = JSON.parse(xhr.responseText);
                    if (resp.success && resp.data && resp.data.html) {
                        content.innerHTML = resp.data.html;
                    }
                } catch (e) {}

                // Update URL
                var params = new URLSearchParams(window.location.search);
                Object.keys(atts).forEach(function (key) {
                    if (atts[key] && key !== 'per_page') {
                        params.set(key, atts[key]);
                    } else {
                        params.delete(key);
                    }
                });
                var newUrl = window.location.pathname + '?' + params.toString();
                history.pushState({ enacast: true, atts: atts }, '', newUrl);
            };

            xhr.onerror = function () {
                container.classList.remove('enacast-loading');
            };

            var body = 'action=enacast_filter&nonce=' + encodeURIComponent(enacast_ajax.nonce)
                + '&shortcode=' + encodeURIComponent(shortcode);
            Object.keys(atts).forEach(function (key) {
                body += '&atts%5B' + encodeURIComponent(key) + '%5D=' + encodeURIComponent(atts[key]);
            });
            xhr.send(body);
        }

        // Handle back button
        window.addEventListener('popstate', function (e) {
            if (e.state && e.state.enacast && e.state.atts) {
                var atts = e.state.atts;
                // Restore form values
                Object.keys(atts).forEach(function (key) {
                    var input = form.querySelector('[name="' + key + '"]');
                    if (input) {
                        if (input.type === 'checkbox') {
                            input.checked = !!atts[key];
                        } else {
                            input.value = atts[key];
                        }
                    }
                });
                fetchFiltered(atts.page || 1);
            }
        });
    }

    // Searchable select dropdowns
    function initSearchableSelects(root) {
        (root || document).querySelectorAll('.enacast-searchable-select').forEach(function (wrapper) {
            var input = wrapper.querySelector('.enacast-searchable-select-input');
            var hidden = wrapper.querySelector('.enacast-searchable-select-value');
            var dropdown = wrapper.querySelector('.enacast-searchable-select-dropdown');
            var options = dropdown.querySelectorAll('.enacast-searchable-select-option');

            input.addEventListener('focus', function () {
                dropdown.classList.add('enacast-searchable-select-open');
                filterOptions('');
            });

            input.addEventListener('input', function () {
                dropdown.classList.add('enacast-searchable-select-open');
                filterOptions(input.value);
                // Clear selection when typing
                hidden.value = '';
            });

            function filterOptions(query) {
                var q = query.toLowerCase();
                options.forEach(function (opt) {
                    var text = opt.textContent.toLowerCase();
                    opt.style.display = (!q || text.indexOf(q) !== -1) ? '' : 'none';
                });
            }

            dropdown.addEventListener('click', function (e) {
                var opt = e.target.closest('.enacast-searchable-select-option');
                if (!opt) return;
                var value = opt.getAttribute('data-value');
                hidden.value = value;
                input.value = value ? opt.textContent.trim() : '';
                dropdown.classList.remove('enacast-searchable-select-open');
            });

            // Close on outside click
            document.addEventListener('click', function (e) {
                if (!wrapper.contains(e.target)) {
                    dropdown.classList.remove('enacast-searchable-select-open');
                }
            });

            // Close on Escape
            input.addEventListener('keydown', function (e) {
                if (e.key === 'Escape') {
                    dropdown.classList.remove('enacast-searchable-select-open');
                    input.blur();
                }
            });
        });
    }

    initSearchableSelects(document);

    // Audio player: marker seek
    document.addEventListener('click', function (e) {
        var btn = e.target.closest('.enacast-marker-time[data-seek]');
        if (!btn) return;

        var seconds = parseInt(btn.getAttribute('data-seek'), 10);
        var player = btn.closest('.enacast-podcast-single');
        if (!player) return;

        var audio = player.querySelector('.enacast-audio');
        if (audio) {
            audio.currentTime = seconds;
            audio.play();
        }
    });

    // Now Playing auto-refresh
    document.querySelectorAll('.enacast-now-playing[data-enacast-refresh]').forEach(function (el) {
        var interval = parseInt(el.getAttribute('data-enacast-refresh'), 10);
        if (interval <= 0) return;

        // Store an ID to re-query the DOM after outerHTML replacement
        var refreshId = 'enacast-np-' + Math.random().toString(36).substr(2, 9);
        el.setAttribute('data-enacast-refresh-id', refreshId);

        setInterval(function () {
            var current = document.querySelector('[data-enacast-refresh-id="' + refreshId + '"]');
            if (!current) return;

            var xhr = new XMLHttpRequest();
            xhr.open('POST', enacast_ajax.ajax_url, true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            xhr.onload = function () {
                try {
                    var resp = JSON.parse(xhr.responseText);
                    if (resp.success && resp.data && resp.data.html) {
                        var tmp = document.createElement('div');
                        tmp.innerHTML = resp.data.html;
                        var newEl = tmp.firstElementChild;
                        if (newEl) {
                            newEl.setAttribute('data-enacast-refresh-id', refreshId);
                            current.replaceWith(newEl);
                        }
                    }
                } catch (e) {}
            };
            xhr.send('action=enacast_filter&nonce=' + encodeURIComponent(enacast_ajax.nonce)
                + '&shortcode=enacast_now_playing');
        }, interval * 1000);
    });
})();
