<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Enacast_API_Client {

    private $api_url;
    private $radio_codename;
    private $cache_ttl;

    private static $instance = null;

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->api_url        = 'https://enacast.com';
        $this->radio_codename = get_option( 'enacast_radio_codename', '' );
        $this->cache_ttl      = (int) get_option( 'enacast_cache_ttl', 300 );
    }

    public function get_radio_codename() {
        return $this->radio_codename;
    }

    /**
     * Make a GET request to the EnaCast API.
     *
     * @param string $endpoint  Relative endpoint (e.g. 'programs/')
     * @param array  $params    Query parameters
     * @param bool   $add_radio Automatically add radio filter
     * @return array|WP_Error   Decoded JSON or WP_Error
     */
    public function get( $endpoint, $params = array(), $add_radio = true ) {
        if ( $add_radio && ! empty( $this->radio_codename ) && ! isset( $params['radio'] ) ) {
            $params['radio'] = $this->radio_codename;
        }

        $url = $this->api_url . '/api/v3/' . ltrim( $endpoint, '/' );
        if ( ! empty( $params ) ) {
            $url = add_query_arg( $params, $url );
        }

        $cache_key = 'enacast_' . md5( $url );

        if ( $this->cache_ttl > 0 ) {
            $cached = get_transient( $cache_key );
            if ( false !== $cached ) {
                return $cached;
            }
        }

        $args = array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        );

        $response = wp_remote_get( $url, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error(
                'enacast_api_error',
                sprintf( __( 'API returned status %d', 'enacast' ), $code ),
                array( 'status' => $code )
            );
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( null === $data ) {
            return new WP_Error( 'enacast_json_error', __( 'Invalid JSON response', 'enacast' ) );
        }

        if ( $this->cache_ttl > 0 ) {
            set_transient( $cache_key, $data, $this->cache_ttl );
        }

        return $data;
    }

    /**
     * Fetch a paginated list.
     */
    public function get_list( $endpoint, $params = array() ) {
        return $this->get( $endpoint, $params );
    }

    /**
     * Fetch a single resource by identifier.
     */
    public function get_detail( $endpoint, $id, $params = array() ) {
        return $this->get( rtrim( $endpoint, '/' ) . '/' . $id . '/', $params );
    }

    // Convenience methods

    public function get_programs( $params = array() ) {
        return $this->get_list( 'programs/', $params );
    }

    public function get_program( $codename ) {
        return $this->get_detail( 'programs', $codename );
    }

    public function get_podcasts( $params = array() ) {
        return $this->get_list( 'podcast/', $params );
    }

    public function get_podcast( $id ) {
        return $this->get_detail( 'podcast', $id );
    }

    public function get_news( $params = array() ) {
        return $this->get_list( 'news/', $params );
    }

    public function get_news_article( $id ) {
        return $this->get_detail( 'news', $id );
    }

    public function get_news_tags( $params = array() ) {
        return $this->get_list( 'news_tags/', $params );
    }

    public function get_schedule_week( $year = null, $week = null ) {
        if ( null === $year ) {
            $year = (int) date( 'o' );
        }
        if ( null === $week ) {
            $week = (int) date( 'W' );
        }
        return $this->get( 'schedules/week/', array(
            'year' => $year,
            'week' => $week,
        ) );
    }

    public function get_agenda( $params = array() ) {
        return $this->get_list( 'agenda/', $params );
    }

    public function get_agenda_tags( $params = array() ) {
        return $this->get_list( 'agenda_tags/', $params );
    }

    public function get_custom_page( $slug ) {
        return $this->get( 'custom_pages/', array( 'slug' => $slug ) );
    }

    public function get_now_playing() {
        return $this->get( 'now_playing' );
    }

    public function get_radio_profile() {
        if ( empty( $this->radio_codename ) ) {
            return new WP_Error( 'enacast_no_codename', __( 'Radio codename not configured', 'enacast' ) );
        }
        return $this->get_detail( 'radios', $this->radio_codename, array(), false );
    }

    public function check_health() {
        return $this->get( 'health', array(), false );
    }

    /**
     * Search via Typesense.
     *
     * @param string $collection Collection name (e.g. 'enacast___news')
     * @param string $query      Search query
     * @param array  $params     Additional Typesense search params
     * @return array|WP_Error    Typesense response or WP_Error
     */
    public function search_typesense( $collection, $query, $params = array() ) {
        $defaults = array(
            'q'        => $query,
            'per_page' => 20,
            'page'     => 1,
        );
        $params = wp_parse_args( $params, $defaults );

        // Always filter by radio
        if ( ! empty( $this->radio_codename ) ) {
            $filter = 'radioCodename:' . $this->radio_codename;
            if ( ! empty( $params['filter_by'] ) ) {
                $params['filter_by'] = $filter . ' && ' . $params['filter_by'];
            } else {
                $params['filter_by'] = $filter;
            }
        }

        $url = rtrim( ENACAST_SEARCH_URL, '/' ) . '/collections/' . $collection . '/documents/search';
        $url = add_query_arg( $params, $url );

        $cache_key = 'enacast_ts_' . md5( $url );
        if ( $this->cache_ttl > 0 ) {
            $cached = get_transient( $cache_key );
            if ( false !== $cached ) {
                return $cached;
            }
        }

        $response = wp_remote_get( $url, array(
            'timeout' => 10,
            'headers' => array(
                'X-TYPESENSE-API-KEY' => ENACAST_SEARCH_API_KEY,
                'Accept'              => 'application/json',
            ),
        ) );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'enacast_search_error', sprintf( __( 'Search returned status %d', 'enacast' ), $code ) );
        }

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( null === $data ) {
            return new WP_Error( 'enacast_search_json_error', __( 'Invalid search response', 'enacast' ) );
        }

        if ( $this->cache_ttl > 0 ) {
            set_transient( $cache_key, $data, $this->cache_ttl );
        }

        return $data;
    }

    /**
     * Search news via Typesense and map results to API format.
     *
     * @param string $query  Search term
     * @param array  $params Extra filters (tag, per_page, page)
     * @return array DRF-compatible response with count and results
     */
    public function search_news( $query, $params = array() ) {
        $ts_params = array(
            'query_by'                => 'headline,body',
            'sort_by'                 => 'datetime_to_publish_ts:desc',
            'highlight_fields'        => 'headline,body',
            'highlight_affix_num_tokens' => 15,
            'per_page'                => isset( $params['per_page'] ) ? (int) $params['per_page'] : 20,
            'page'                    => isset( $params['page'] ) ? (int) $params['page'] : 1,
        );

        if ( ! empty( $params['tag'] ) ) {
            $ts_params['filter_by'] = 'tags:=' . $params['tag'];
        }

        $response = $this->search_typesense( 'enacast___news', $query, $ts_params );
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $articles = array();
        if ( ! empty( $response['hits'] ) ) {
            foreach ( $response['hits'] as $hit ) {
                $doc = $hit['document'];
                $short_body = ! empty( $doc['body'] ) ? wp_strip_all_tags( $doc['body'] ) : '';
                $short_body = preg_replace( '/\s+/', ' ', $short_body );
                if ( strlen( $short_body ) > 200 ) {
                    $short_body = substr( $short_body, 0, 200 ) . '...';
                }

                $tags = array();
                if ( ! empty( $doc['tags'] ) && is_array( $doc['tags'] ) ) {
                    foreach ( $doc['tags'] as $tag_slug ) {
                        $tags[] = array(
                            'title' => ucfirst( str_replace( '-', ' ', $tag_slug ) ),
                            'slug'  => $tag_slug,
                        );
                    }
                }

                // Construct full public ID: {radioCodename}_news_{_public_id}
                $public_id = '';
                if ( ! empty( $doc['_public_id'] ) ) {
                    $radio = ! empty( $doc['radioCodename'] ) ? $doc['radioCodename'] : $this->radio_codename;
                    $public_id = $radio . '_news_' . $doc['_public_id'];
                } else {
                    $public_id = $doc['id'];
                }

                $articles[] = array(
                    'id'                          => $public_id,
                    'slug'                        => isset( $doc['slug'] ) ? $doc['slug'] : '',
                    'headline'                    => isset( $doc['headline'] ) ? wp_strip_all_tags( $doc['headline'] ) : '',
                    'sub_headline'                => isset( $doc['sub_headline'] ) ? $doc['sub_headline'] : '',
                    'unformatted_short_body'      => $short_body,
                    'main_image_cropped_landscape' => isset( $doc['main_image'] ) ? $doc['main_image'] : '',
                    'created'                     => isset( $doc['datetime_to_publish_ts'] ) ? date( 'c', $doc['datetime_to_publish_ts'] ) : '',
                    'pinned'                      => ! empty( $doc['pinned'] ),
                    'tags'                        => $tags,
                );
            }
        }

        return array(
            'count'   => isset( $response['found'] ) ? (int) $response['found'] : count( $articles ),
            'results' => $articles,
        );
    }

    /**
     * Search podcasts via Typesense and map results to API format.
     *
     * @param string $query  Search term
     * @param array  $params Extra filters (program, per_page, page)
     * @return array DRF-compatible response with count and results
     */
    public function search_podcasts( $query, $params = array() ) {
        $ts_params = array(
            'query_by'          => 'podcastTitle,podcastDescription',
            'sort_by'           => 'recorded_ts:desc',
            'highlight_fields'  => 'podcastTitle,podcastDescription',
            'snippet_threshold' => 30,
            'per_page'          => isset( $params['per_page'] ) ? (int) $params['per_page'] : 20,
            'page'              => isset( $params['page'] ) ? (int) $params['page'] : 1,
        );

        if ( ! empty( $params['program'] ) ) {
            $ts_params['filter_by'] = 'programCodename:' . $params['program'];
        }

        $response = $this->search_typesense( 'enacast___podcasts', $query, $ts_params );
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $podcasts = array();
        if ( ! empty( $response['hits'] ) ) {
            foreach ( $response['hits'] as $hit ) {
                $doc = $hit['document'];
                $podcasts[] = array(
                    'id'                         => isset( $doc['podcastPublicID'] ) ? $doc['podcastPublicID'] : ( isset( $doc['publicID'] ) ? $doc['publicID'] : $doc['id'] ),
                    'name'                       => isset( $doc['podcastTitle'] ) ? $doc['podcastTitle'] : '',
                    'program_name'               => isset( $doc['programTitle'] ) ? $doc['programTitle'] : ( isset( $doc['programName'] ) ? $doc['programName'] : '' ),
                    'program_codename'           => isset( $doc['programCodename'] ) ? $doc['programCodename'] : '',
                    'datetime_tz'                => isset( $doc['recorded_ts'] ) ? date( 'c', $doc['recorded_ts'] ) : '',
                    'duration_in_seconds_cutted' => isset( $doc['duration'] ) ? (int) $doc['duration'] : 0,
                    'logo_url'                   => isset( $doc['thumbnail'] ) ? $doc['thumbnail'] : '',
                    'program_logo_400x400'       => isset( $doc['image'] ) ? $doc['image'] : '',
                );
            }
        }

        return array(
            'count'   => isset( $response['found'] ) ? (int) $response['found'] : count( $podcasts ),
            'results' => $podcasts,
        );
    }

    /**
     * Flush all EnaCast transient caches.
     */
    public static function flush_cache() {
        global $wpdb;
        $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_enacast\_%' OR option_name LIKE '_transient_timeout_enacast\_%'"
        );
    }
}
