<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Enacast_Settings {

    public function init() {
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_filter( 'plugin_action_links_' . ENACAST_PLUGIN_BASENAME, array( $this, 'settings_link' ) );
        add_action( 'wp_ajax_enacast_test_connection', array( $this, 'ajax_test_connection' ) );
        add_action( 'update_option_enacast_radio_codename', array( $this, 'on_settings_change' ) );
    }

    public function add_settings_page() {
        add_options_page(
            __( 'EnaCast Settings', 'enacast' ),
            __( 'EnaCast', 'enacast' ),
            'manage_options',
            'enacast',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting( 'enacast_settings', 'enacast_radio_codename', array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        ) );
        register_setting( 'enacast_settings', 'enacast_cache_ttl', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 300,
        ) );
        add_settings_section(
            'enacast_main',
            __( 'API Configuration', 'enacast' ),
            null,
            'enacast'
        );

        add_settings_field( 'enacast_radio_codename', __( 'Radio Codename', 'enacast' ), array( $this, 'field_radio_codename' ), 'enacast', 'enacast_main' );
        add_settings_field( 'enacast_cache_ttl', __( 'Cache TTL (seconds)', 'enacast' ), array( $this, 'field_cache_ttl' ), 'enacast', 'enacast_main' );

        // Detail page settings
        register_setting( 'enacast_settings', 'enacast_page_program', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );
        register_setting( 'enacast_settings', 'enacast_page_podcast', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );
        register_setting( 'enacast_settings', 'enacast_page_news_article', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );
        register_setting( 'enacast_settings', 'enacast_page_schedule', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );
        register_setting( 'enacast_settings', 'enacast_page_news', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );
        register_setting( 'enacast_settings', 'enacast_page_podcasts', array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 0,
        ) );

        add_settings_section(
            'enacast_pages',
            __( 'Detail Pages', 'enacast' ),
            array( $this, 'section_pages_description' ),
            'enacast'
        );

        add_settings_field( 'enacast_page_program', __( 'Program detail page', 'enacast' ), array( $this, 'field_page_program' ), 'enacast', 'enacast_pages' );
        add_settings_field( 'enacast_page_podcast', __( 'Podcast detail page', 'enacast' ), array( $this, 'field_page_podcast' ), 'enacast', 'enacast_pages' );
        add_settings_field( 'enacast_page_news_article', __( 'News article page', 'enacast' ), array( $this, 'field_page_news_article' ), 'enacast', 'enacast_pages' );
        add_settings_field( 'enacast_page_schedule', __( 'Schedule page', 'enacast' ), array( $this, 'field_page_schedule' ), 'enacast', 'enacast_pages' );
        add_settings_field( 'enacast_page_news', __( 'News list page', 'enacast' ), array( $this, 'field_page_news' ), 'enacast', 'enacast_pages' );
        add_settings_field( 'enacast_page_podcasts', __( 'Podcasts list page', 'enacast' ), array( $this, 'field_page_podcasts' ), 'enacast', 'enacast_pages' );

        // Subscribe buttons settings
        foreach ( array_keys( self::get_subscribe_platforms() ) as $key ) {
            register_setting( 'enacast_settings', 'enacast_subscribe_' . $key, array(
                'type'              => 'string',
                'sanitize_callback' => array( __CLASS__, 'sanitize_checkbox' ),
                'default'           => '1',
            ) );
        }

        add_settings_section(
            'enacast_subscribe',
            __( 'Subscribe buttons', 'enacast' ),
            array( $this, 'section_subscribe_description' ),
            'enacast'
        );

        add_settings_field( 'enacast_subscribe_buttons', __( 'Enabled platforms', 'enacast' ), array( $this, 'field_subscribe_buttons' ), 'enacast', 'enacast_subscribe' );
    }

    public static function get_subscribe_platforms() {
        return array(
            'rss'         => 'RSS',
            'apple'       => 'Apple Podcasts',
            'overcast'    => 'Overcast',
            'pocketcasts' => 'Pocket Casts',
            'castro'      => 'Castro',
        );
    }

    public static function sanitize_checkbox( $value ) {
        return $value ? '1' : '0';
    }

    public static function is_subscribe_enabled( $key ) {
        return get_option( 'enacast_subscribe_' . $key, '1' ) === '1';
    }

    public function section_subscribe_description() {
        echo '<p>' . esc_html__( 'Choose which subscribe buttons to show on program detail pages.', 'enacast' ) . '</p>';
    }

    public function field_subscribe_buttons() {
        $platforms = self::get_subscribe_platforms();
        foreach ( $platforms as $key => $label ) {
            $option  = 'enacast_subscribe_' . $key;
            $checked = get_option( $option, '1' ) === '1';
            echo '<label style="display:block;margin-bottom:0.4em;">';
            echo '<input type="hidden" name="' . esc_attr( $option ) . '" value="0" />';
            echo '<input type="checkbox" name="' . esc_attr( $option ) . '" value="1" ' . checked( $checked, true, false ) . ' /> ';
            echo esc_html( $label );
            echo '</label>';
        }
    }

    public function section_pages_description() {
        echo '<p>' . esc_html__( 'Select the pages that contain the detail shortcodes. Links from list views will point to these pages.', 'enacast' ) . '</p>';
    }

    public function field_radio_codename() {
        $value = get_option( 'enacast_radio_codename', '' );
        echo '<input type="text" name="enacast_radio_codename" value="' . esc_attr( $value ) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__( 'The codename of your radio station in EnaCast', 'enacast' ) . '</p>';
    }

    public function field_cache_ttl() {
        $value = get_option( 'enacast_cache_ttl', 300 );
        echo '<input type="number" name="enacast_cache_ttl" value="' . esc_attr( $value ) . '" min="0" step="1" class="small-text" />';
        echo '<p class="description">' . esc_html__( 'How long to cache API responses (0 to disable)', 'enacast' ) . '</p>';
    }

    public function field_page_program() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_program',
            'selected'          => get_option( 'enacast_page_program', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_program] shortcode', 'enacast' ) . '</p>';
    }

    public function field_page_podcast() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_podcast',
            'selected'          => get_option( 'enacast_page_podcast', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_podcast] shortcode', 'enacast' ) . '</p>';
    }

    public function field_page_news_article() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_news_article',
            'selected'          => get_option( 'enacast_page_news_article', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_news_article] shortcode', 'enacast' ) . '</p>';
    }

    public function field_page_schedule() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_schedule',
            'selected'          => get_option( 'enacast_page_schedule', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_schedule] shortcode', 'enacast' ) . '</p>';
    }

    public function field_page_news() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_news',
            'selected'          => get_option( 'enacast_page_news', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_news] shortcode', 'enacast' ) . '</p>';
    }

    public function field_page_podcasts() {
        wp_dropdown_pages( array(
            'name'              => 'enacast_page_podcasts',
            'selected'          => get_option( 'enacast_page_podcasts', 0 ),
            'show_option_none'  => __( '— Select —', 'enacast' ),
            'option_none_value' => 0,
        ) );
        echo '<p class="description">' . esc_html__( 'Page with [enacast_podcasts] shortcode', 'enacast' ) . '</p>';
    }

    /**
     * Get the URL for a detail page with query parameter.
     *
     * Falls back to relative ?param=value if no page is configured.
     */
    public static function get_detail_url( $type, $param, $value ) {
        $option_map = array(
            'program'      => 'enacast_page_program',
            'podcast'      => 'enacast_page_podcast',
            'news_article' => 'enacast_page_news_article',
        );

        if ( ! isset( $option_map[ $type ] ) ) {
            return '?' . urlencode( $param ) . '=' . urlencode( $value );
        }

        $page_id = (int) get_option( $option_map[ $type ], 0 );
        if ( $page_id > 0 ) {
            return add_query_arg( $param, $value, get_permalink( $page_id ) );
        }

        return '?' . urlencode( $param ) . '=' . urlencode( $value );
    }

    /**
     * Get the permalink for a configured page (no query params).
     */
    public static function get_page_url( $type ) {
        $page_id = (int) get_option( 'enacast_page_' . $type, 0 );
        if ( $page_id > 0 ) {
            return get_permalink( $page_id );
        }
        return '';
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        include ENACAST_PLUGIN_DIR . 'templates/admin/settings-page.php';
    }

    public function settings_link( $links ) {
        $url  = admin_url( 'options-general.php?page=enacast' );
        $link = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Settings', 'enacast' ) . '</a>';
        array_unshift( $links, $link );
        return $links;
    }

    public function on_settings_change() {
        Enacast_API_Client::flush_cache();
    }

    public function ajax_test_connection() {
        check_ajax_referer( 'enacast_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorized', 'enacast' ) );
        }

        $client = Enacast_API_Client::instance();

        // Test health endpoint
        $health = $client->check_health();
        if ( is_wp_error( $health ) ) {
            wp_send_json_error( sprintf(
                __( 'Health check failed: %s', 'enacast' ),
                $health->get_error_message()
            ) );
        }

        // Test radio endpoint
        $radio = $client->get_radio_profile();
        if ( is_wp_error( $radio ) ) {
            wp_send_json_error( sprintf(
                __( 'Radio lookup failed: %s', 'enacast' ),
                $radio->get_error_message()
            ) );
        }

        $name = isset( $radio['name'] ) ? $radio['name'] : '';
        wp_send_json_success( sprintf(
            __( 'Connected successfully! Radio: %s', 'enacast' ),
            $name
        ) );
    }
}
